/**
 * \file: caam.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * \component: Secure Data Container
 * \brief : Implementation of common architecure specific functions
 *
 * \author: Ian Molton (ian.molton@codethink.co.uk)
 *      Norbert Uetrecht (nuetrecht@de.adit-jv.com)
 *      Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2014 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <string.h>

#include <sdc.h>
#include <private/sdc_advanced.h>
#include <private/sdc_arch.h>
#include <linux/caam_ee.h>
#include "caam.h"
#include <sdc/arch/errors.h>

/* used to define that the library expects a certain
 * version-subversion for correct operation
 */
#define LIBSDC_FEATURE_VERSION      12
#define LIBSDC_FEATURE_SUBVERSION   0


sdc_error_t error_from_open_errno(int err)
{
    switch (err) {
    case EPERM:
        return SDC_ACCESS_DENIED;
    case EACCES:
        return SDC_ACCESS_DENIED;
    default:
        return SDC_IMX6_CAAMEE_DEVICENODE_IOCTL_FAILED;
    }
}

sdc_error_t error_from_ioctl_errno(int err, unsigned long int ioctl_cmd, uint32_t ioctl_flags)
{
    switch (err) {
    case ENOKEY:
        switch(ioctl_cmd) {
        /* for IOCTLs trying to load a key this means the key does not exist */
        case CAAM_SET_SESSION_KEYSTORE_KEY:
        case CAAM_SET_SESSION_BUILTIN_KEY:
            return SDC_KID_NOT_AVAILABLE;
        default:
            /* for others the key is just not loaded yet */
            return SDC_KEY_UNSET;
        }
    case EEXIST:
        return SDC_KID_EXISTS;
    case EPERM:
        return SDC_ACCESS_DENIED;
    case EACCES:
        return SDC_ACCESS_DENIED;
    case EKEYREVOKED:
        return SDC_KEY_TAMPERED;
    case EOPNOTSUPP:
        switch (ioctl_cmd) {
        case CAAM_ENCRYPT_DATA:
        case CAAM_DECRYPT_DATA:
        case CAAM_WRAP_DATA:
        case CAAM_UNWRAP_DATA:
        case CAAM_SIGN_DATA:
        case CAAM_VERIFY_DATA:
        case CAAM_DGST_DATA:
        case CAAM_ASYM_ENCRYPT_DATA:
        case CAAM_ASYM_DECRYPT_DATA:
        case CAAM_ASYM_SIGN_HASH:
        case CAAM_ASYM_VERIFY_HASH:
            /* the key length is not supported */
            return SDC_ALG_MODE_INVALID;
        default:
            return SDC_OP_NOT_SUPPORTED;
        }
    case EAGAIN:
        return SDC_KEY_LOCKED;
    case EKEYREJECTED:
        return SDC_KEY_NOT_READY;
    case EBADMSG:
        return SDC_AUTHENTICATION_FAILED;
    case EBADF:
        return SDC_IMX6_INVALID_DEVICENODE;
    case EINVAL:
        /*
         * for asymmetric decrypt(one shot), verify-hash(one shot) or verify(at final)
         * EINVAL will occur in case the key used for encrypt or
         * sign is belonging to a different key pair
         */
        if (
            /* asymmetric decrypt - only one shot supported by driver */
            (ioctl_cmd == CAAM_ASYM_DECRYPT_DATA) ||
            /* asymmetric verify(precomputed hash) - only one shot supported by driver */
            (ioctl_cmd == CAAM_ASYM_VERIFY_HASH) ||
            /* asymmetric verify (ALG = RSA PKCS1V15) - finals (i.e. either single or multi finalize) */
            ((ioctl_cmd == CAAM_VERIFY_DATA) &&
             ((ioctl_flags & CAAM_EE_SIGN_ALG_MASK) == CAAM_EE_SIGN_ALG_RSA_PKCS1V15_FLAG) &&
             (((ioctl_flags & CAAM_OP_MASK) == CAAM_OP_SINGLE) ||
              ((ioctl_flags & CAAM_OP_MASK) == CAAM_OP_MULTI_FINALIZE)))
            )
            return SDC_AUTHENTICATION_FAILED;
    /* Fall through */
    default:
        return SDC_IMX6_CAAMEE_DEVICENODE_IOCTL_FAILED;
    }
}


#define VERSION_INFO_STR_MAX 256

/* External available interface as defined in sdc.h */
#ifdef CAAM_GET_KERNEL_VERSION
sdc_error_t sdc_arch_kernel_version_verify(char **version_info_str)
{
    int fd;
    int res;
    sdc_error_t err = SDC_OK;
    struct caam_ee_version kernel_version_info;
    uint32_t sw_feature_version;
    uint32_t sw_feature_subversion;

    if (version_info_str != NULL) {
        *version_info_str = NULL;
    }

    fd = open(CAAM_DEVICE, O_RDONLY, 0);

    if (fd < 0) {
        fprintf(stderr,"Error: Couldn't open device\n");
        return error_from_open_errno(errno);
    }

    memset(&kernel_version_info, 0, sizeof(kernel_version_info));

    res = ioctl(fd, CAAM_GET_KERNEL_VERSION, &kernel_version_info);
    if (res != 0) {
        /* kernel does not provide version info - initialize to 0 */
        kernel_version_info.version = 0;
        kernel_version_info.subversion = 0;
        kernel_version_info.ioctl_comp_version = 0;
    }

    /* version required by library */
    sw_feature_version = LIBSDC_FEATURE_VERSION;
    sw_feature_subversion = LIBSDC_FEATURE_SUBVERSION;

    /* generate info string */
    if (version_info_str != NULL) {
        *version_info_str = malloc(VERSION_INFO_STR_MAX);

        if (*version_info_str) {
            snprintf(*version_info_str, VERSION_INFO_STR_MAX,
                     "libSDC kernel %u-%u ioctl-version %u, library %u-%u",
                     kernel_version_info.version,
                     kernel_version_info.subversion,
                     kernel_version_info.ioctl_comp_version,
                     sw_feature_version, sw_feature_subversion);
        }
    }

    /* check version info
     * the kernel has to provide at least the amount of features
     * as available when building/commiting the library
     *     version > sw_feature_version
     *     version == sw_feature_version && subversion >= sw_feature_subversion
     *
     * The ioctls need to be compatible to ioctls
     * available when building the library
     *  ioctl_comp_version <= sw_feature_version
     */
    if ((kernel_version_info.version < sw_feature_version) ||
        ((kernel_version_info.version == sw_feature_version) &&
         (kernel_version_info.subversion < sw_feature_subversion)) ||
        (kernel_version_info.ioctl_comp_version > sw_feature_version)) {
        err = SDC_INVALID_VERSION;
    }

    close(fd);

    return err;
}
#else
sdc_error_t sdc_arch_kernel_version_verify(char **version_info_str)
{
    if (version_info_str != NULL) {
        *version_info_str = malloc(VERSION_INFO_STR_MAX);

        if (*version_info_str) {
            snprintf(*version_info_str, VERSION_INFO_STR_MAX,
                     "libSDC version check not active");
        }
    }

    return SDC_OK;
}
#endif

void caam_keylen_bmsk_to_sdc(uint32_t caam_bmsk, sdc_key_len_bmsk_t *bmsk, sdc_key_len_t *max_len)
{
    sdc_key_len_bmsk_t tmp_bmsk = SDC_KEY_LEN_BMSK_NONE;
    sdc_key_len_t tmp_max = SDC_KEY_LEN_UNKNOWN;

    if (caam_bmsk & CAAM_KEY_LEN_64BIT_BMSK) {
        tmp_bmsk |= SDC_KEY_LEN_BMSK_64bit;
        tmp_max = SDC_KEY_LEN_64bit;
    }
    if (caam_bmsk & CAAM_KEY_LEN_128BIT_BMSK) {
        tmp_bmsk |= SDC_KEY_LEN_BMSK_128bit;
        tmp_max = SDC_KEY_LEN_128bit;
    }
    if (caam_bmsk & CAAM_KEY_LEN_192BIT_BMSK) {
        tmp_bmsk |= SDC_KEY_LEN_BMSK_192bit;
        tmp_max = SDC_KEY_LEN_192bit;
    }
    if (caam_bmsk & CAAM_KEY_LEN_256BIT_BMSK) {
        tmp_bmsk |= SDC_KEY_LEN_BMSK_256bit;
        tmp_max = SDC_KEY_LEN_256bit;
    }
    if (caam_bmsk & CAAM_KEY_LEN_1024BIT_BMSK) {
        tmp_bmsk |= SDC_KEY_LEN_BMSK_1024bit;
        tmp_max = SDC_KEY_LEN_1024bit;
    }
    if (caam_bmsk & CAAM_KEY_LEN_2048BIT_BMSK) {
        tmp_bmsk |= SDC_KEY_LEN_BMSK_2048bit;
        tmp_max = SDC_KEY_LEN_2048bit;
    }
    if (caam_bmsk & CAAM_KEY_LEN_4096BIT_BMSK) {
        tmp_bmsk |= SDC_KEY_LEN_BMSK_4096bit;
        tmp_max = SDC_KEY_LEN_4096bit;
    }

    if (bmsk)
        *bmsk = tmp_bmsk;

    if (max_len)
        *max_len = tmp_max;
}

sdc_error_t sdc_arch_key_len_bmsks(sdc_key_fmt_t key_fmt, sdc_key_len_bmsk_t *arch_keylen_bmsk)
{
    switch (key_fmt) {
    case SDC_KEY_FMT_SIMPLE_SYM:
        *arch_keylen_bmsk = SDC_KEY_LEN_BMSK_64bit | SDC_KEY_LEN_BMSK_128bit |
                            SDC_KEY_LEN_BMSK_192bit | SDC_KEY_LEN_BMSK_256bit;
        break;
    case SDC_KEY_FMT_RSA_PUBLIC:
    case SDC_KEY_FMT_RSA_PRIVATE:
        *arch_keylen_bmsk = SDC_KEY_LEN_BMSK_1024bit | SDC_KEY_LEN_BMSK_2048bit |
                            SDC_KEY_LEN_BMSK_4096bit;
        break;
    default:
        /* unsupported - don't support anything */
        *arch_keylen_bmsk = SDC_KEY_LEN_BMSK_NONE;
    }

    return SDC_OK;
}

int caam_perform_encrypt_decrypt_ioctl_operation(sdc_session_t *session, struct caam_ee_encrypt_decrypt_params *iodata,
                                                 const uint8_t *in_data_ptr, uint8_t *out_data_ptr,
                                                 size_t in_len, size_t out_len, uint32_t iodata_flags)
{

    if (iodata) {
        iodata->flags = iodata_flags;
        iodata->in = (uint8_t *) in_data_ptr;
        iodata->out = out_data_ptr;
        iodata->in_len = in_len;
        iodata->out_len = out_len;
    }

    return ioctl(session->arch.fd, session->arch.request, iodata);
}

int caam_perform_sign_verify_ioctl_operation(sdc_session_t *session, struct caam_ee_sign_verify_params *iodata,
                                             const uint8_t *in_data_ptr, size_t in_len,
                                             uint8_t *tag_out_data, size_t tag_out_data_len,
                                             const uint8_t *tag_in_data, size_t tag_in_data_len,
                                             uint32_t iodata_flags)
{

    if (iodata) {
        iodata->flags = iodata_flags;
        iodata->in = (uint8_t *) in_data_ptr;
        if (session->arch.request == CAAM_SIGN_DATA) {
            iodata->tag = (uint8_t *) tag_out_data;
            iodata->tag_len = tag_out_data_len;
        } else {
            iodata->tag = (uint8_t *) tag_in_data;
            iodata->tag_len = tag_in_data_len;
        }
        iodata->in_len = in_len;
    }

    return ioctl(session->arch.fd, session->arch.request, iodata);
}

sdc_error_t caam_perform_abort_operation(sdc_session_t *session)
{
    sdc_error_t err = SDC_OK;
    bool retry = true;
    int res;
    int res_errno;

    /* cancel not completed operations */
    session->arch.request = CAAM_ABORT_OPERATION;
    while (retry) {
        res = caam_perform_encrypt_decrypt_ioctl_operation(session, NULL, NULL, NULL, 0, 0, 0);
        res_errno = errno;

        if (res == 0)
            break;

        if (res_errno == EINVAL) { /* No operation is in progress */
            break;
        } else if (res_errno == EAGAIN) {
            /* retry in case of EAGAIN */
            continue;
        } else {
            err = error_from_ioctl_errno(res_errno, CAAM_ABORT_OPERATION, 0);
            retry = false;
        }
    }

    return err;
}

int caam_perform_dgst_ioctl_operation(sdc_session_t *session, struct caam_ee_dgst_params *iodata,
                                      const uint8_t *in_data_ptr, size_t in_len,
                                      uint8_t *dgst_out_data, size_t dgst_out_data_len,
                                      uint32_t iodata_flags)
{

    if (iodata) {
        iodata->flags = iodata_flags;
        iodata->in = (uint8_t *) in_data_ptr;
        iodata->in_len = in_len;
        iodata->dgst = (uint8_t *) dgst_out_data;
        iodata->dgst_len = dgst_out_data_len;
    }

    return ioctl(session->arch.fd, session->arch.request, iodata);
}
bool caam_is_op_in_progress(sdc_session_t *session)
{
    /* operation finished successfully if sequence state is SDC_NO_OP */
    if((session->sequence.op_state == SDC_NO_OP))
        return false;

    return true;
}

int caam_perform_wrap_unwrap_ioctl_operation(sdc_session_t *session, struct caam_ee_wrap_unwrap_params *iodata,
                                             const uint8_t *in_data, const size_t in_data_len,
                                             uint8_t *out_tag_data, size_t out_tag_data_len,
                                             const uint8_t *in_tag_data, size_t in_tag_data_len,
                                             uint8_t *out_data_ptr, size_t out_data_len,
                                             uint32_t iodata_flags)
{
    if (iodata) {
        iodata->flags = iodata_flags;
        iodata->in = (uint8_t *) in_data;
        iodata->out = out_data_ptr;
        iodata->in_len = in_data_len;
        iodata->out_len = out_data_len;
        if (session->arch.request == CAAM_WRAP_DATA) {
            iodata->tag = (uint8_t *) out_tag_data;
            iodata->tag_len = out_tag_data_len;
            iodata->desired_tag_len = out_tag_data_len;
        } else {
            iodata->tag = (uint8_t *) in_tag_data;
            iodata->tag_len = in_tag_data_len;
            iodata->desired_tag_len = in_tag_data_len;
        }
    }
    return ioctl(session->arch.fd, session->arch.request, iodata);
}
